﻿﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using AHOBPR_Job_Runner.DasXml;
using System.Security.Cryptography.X509Certificates;

namespace AHOBPR_Job_Runner
{
    public class DasExportJob : LoggedJob
    {
        protected override String LogFileName { get { return "AHOBPR_Das_Export.log"; } }
        public override string Id { get { return "DE"; } }
        List<Questionnaire> Questionnaires;

        public DasExportJob()
        {
            Questionnaires = new List<Questionnaire>();
        }

        public override void Run()
        {
            base.StartLog();
            LogMessageToFile("Starting AHOBPR DAS Export...");
            base.CheckDiskSpace();

            GetAllUntransferredRegistrantIds();
            //GetPdfs();
            //SendPdfsToDas();

            LogMessageToFile("AHOBPR DAS Export execution is completed.");
        }

        private void GetAllUntransferredRegistrantIds()
        {
            //  using (SqlConnection dbConnection = new SqlConnection(System.Configuration.ConfigurationSettings.AppSettings["SQLServerConnectionString"]))
            using (SqlConnection dbConnection = new SqlConnection(System.Configuration.ConfigurationManager.AppSettings["SQLServerConnectionString"]))
            {
                try
                {
                    dbConnection.Open();
                    if (dbConnection.State == ConnectionState.Open)
                    {
                        LogMessageToFile("The SqlConnection is open.");
                    }
                    else
                    {
                        LogMessageToFile("ERROR: Unabale to open a connection to SQL.");
                        return;
                    }

                    LogMessageToFile("Gathering List Of Registrant IDs that haven't sent PDF's...");
                    string sqlText = "select r.registrant_id, r.user_id \r\n " +
                                    "from ahobpr.registrant r \r\n " +
                                    "join ahobpr.registrant_file p on r.user_id = p.file_id \r\n " +
                                    "where not exists \r\n " +
                                    "(select x.registrant_id \r\n " +
                                    "from ahobpr.INTERFACE_CALL_LOG x \r\n " +
                                    "where r.REGISTRANT_ID = x.REGISTRANT_ID AND x.INTERFACE_NAME = 'DAS') ";

                    SqlCommand sqlCmd = new SqlCommand(sqlText, dbConnection);
                    try
                    {
                        using (SqlDataReader reader = sqlCmd.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                Questionnaire q = new Questionnaire();
                                q.RegistrantId = reader[0].ToString();
                                q.RegistrantUserId = reader[1].ToString();
                                Questionnaires.Add(q);
                            }
                        }
                        LogMessageToFile("Gathered " + Questionnaires.Count.ToString() + " ids.");
                    }
                    catch (Exception e)
                    {
                        LogErrorMessage(e);
                    }

                    // Process queue
                    int totalRecords = 0;
                    LogMessageToFile("Populating DAS XML for each ID...");
                    foreach (Questionnaire questionnaire in Questionnaires)
                    {
                        sqlText = "select r.SSN, r.FIRST_NAME, r.MIDDLE_NAME, r.LAST_NAME, r.BIRTH_DATE, r.GENDER, r.EDIPI, \r\n " +
                            "f.form_response_id,  b.BRANCH_OF_SERVICE,  \r\n " +
                             "  (select top 1 c.BRANCH_OF_SERVICE \r\n " +
                             "  from AHOBPR.REGISTRANT_DEPLOYMENT d  \r\n " +
                             "  left join ahobpr.STD_BRANCH_OF_SERVICE c on d.STD_BRANCH_OF_SERVICE_ID = c.STD_BRANCH_OF_SERVICE_ID \r\n " +
                             "  where d.REGISTRANT_ID = r.REGISTRANT_ID  \r\n " +
                             "  and d.IS_ELIGIBLE_FLAG = 1) \r\n " +
                            "from ahobpr.registrant r \r\n " +
                            "join ahobpr.form_response f on r.registrant_id = f.registrant_id \r\n " +
                            "left join ahobpr.STD_BRANCH_OF_SERVICE b on r.STD_BRANCH_OF_SERVICE_ID_CURRENT = b.STD_BRANCH_OF_SERVICE_ID \r\n " +
                            "where r.REGISTRANT_ID = @registrantId";
                        sqlCmd = new SqlCommand(sqlText, dbConnection);
                        try
                        {
                            sqlCmd.Parameters.Add(new SqlParameter("@registrantId", questionnaire.RegistrantId));
                            using (SqlDataReader dataReader = sqlCmd.ExecuteReader())
                            {
                                while (dataReader.Read())
                                {
                                    questionnaire.Person.Ssn = dataReader[0].ToString();
                                    questionnaire.Person.GivenName = dataReader[1].ToString();
                                    questionnaire.Person.MiddleName = dataReader[2].ToString();
                                    questionnaire.Person.Surname = dataReader[3].ToString();
                                    questionnaire.Person.BirthDate = dataReader[4] == null ? (DateTime?)null : dataReader[4] == DBNull.Value ? (DateTime?)null : DateTime.Parse(dataReader[4].ToString());
                                    questionnaire.Person.Sex = dataReader[5].ToString();
                                    questionnaire.Client.Edipi = dataReader[6].ToString();
                                    questionnaire.Doc.AhobprId.Id = dataReader[7].ToString();
                                    questionnaire.Person.CurrentMilitarySummary.MilitaryBranchName = dataReader[8] == DBNull.Value ? "" : dataReader[8].ToString();
                                    questionnaire.Person.QualifyingMilitarySummary.MilitaryBranchName = dataReader[9] == DBNull.Value ? "" : dataReader[9].ToString();
                                    questionnaire.Facility = new Facility();

                                    totalRecords++;
                                }
                            }
                        }
                        catch (Exception e)
                        {
                            LogErrorMessage(e);
                            LogMessageToFile("AHOBPR DAS Export: unable to load XML data for registrant ID = " + questionnaire.RegistrantId);
                        }
                    }

                    dbConnection.Close();
                    LogMessageToFile("Created " + totalRecords + " XML strings.");
                    LogMessageToFile("The SqlConnection is closed.");
                }
                catch (Exception ex)
                {
                    LogErrorMessage(ex);
                    LogMessageToFile("AHOBPR DAS Export execution unsuccessful!");
                    return;
                }
            }
        }

        private void GetPdfs()
        {
            LogMessageToFile("AHOBPR DAS Export collecting PDFs.");
            foreach (Questionnaire questionnaire in Questionnaires)
            {
                try
                {
                    String secretKey = System.Configuration.ConfigurationManager.AppSettings["secretKey"];
                    HttpWebRequest getRequest = (HttpWebRequest)WebRequest.Create(System.Configuration.ConfigurationManager.AppSettings["RestServiceUrl"] + "/download/responsePdf/" + questionnaire.RegistrantUserId);
                    getRequest.ContentType = "application/json";
                    getRequest.Accept = "application/json";
                    getRequest.Method = "GET";
                    getRequest.Headers.Add("Authorization", "Basic " + secretKey);

                    HttpWebResponse getResponse = (HttpWebResponse)getRequest.GetResponse();
                    String result = new StreamReader(getResponse.GetResponseStream()).ReadToEnd();
                    int startIndex = result.IndexOf(":[");
                    if (0 < startIndex)
                    {
                        result = result.Substring(startIndex + 2, result.Length - 2 - (startIndex + 2));
                        String[] tokens = result.Split(new char[] { ',' });
                        byte[] bites = new byte[tokens.Length];
                        for (int i = 0; i < tokens.Length; i++)
                        {
                            bites[i] = byte.Parse(tokens[i]);
                        }
                        questionnaire.Attachments.Binary = bites;
                    }
                }
                catch (Exception ex)
                {
                    LogMessageToFile("AHOBPR DAS Export: Unable to get PDF for registrant ID = " + questionnaire.RegistrantId);
                    questionnaire.Attachments.Binary = null;
                }
            }
            LogMessageToFile("AHOBPR DAS Export finished collecting PDFs.");
        }

        private void SendPdfsToDas()
        {
            foreach (Questionnaire questionnaire in Questionnaires)
            {
                if (questionnaire.Attachments.Binary != null)
                {
                    SendAPdfToDas(questionnaire);
                }
            }
        }

        private void SendAPdfToDas(Questionnaire questionnaire)
        {
            HttpRequestMessage requestMessage = new HttpRequestMessage(HttpMethod.Post, new Uri(System.Configuration.ConfigurationManager.AppSettings["DasUri"]));
            requestMessage.Headers.ExpectContinue = false;
            MultipartFormDataContent multiPartContent = new MultipartFormDataContent();
            ByteArrayContent byteArrayContent = new ByteArrayContent(System.Text.Encoding.UTF8.GetBytes(questionnaire.Xml));
            byteArrayContent.Headers.Add("Content-Type", "text/xml");
            multiPartContent.Add(byteArrayContent, "\"file\"");
            requestMessage.Content = multiPartContent;

            //Add Certificates
            WebRequestHandler handler = new WebRequestHandler();
            X509Certificate certificate = GetCert(System.Configuration.ConfigurationManager.AppSettings["AhobprDasCert"]);
            if (certificate == null)
            {
                LogMessageToFile("AHOBPR DAS Export: Unable to load certificate = " + System.Configuration.ConfigurationManager.AppSettings["AhobprDasCert"]);
            }
            else
            {
                handler.ClientCertificates.Add(certificate);
            }
            HttpClient httpClient = new HttpClient(handler);

            String json = "";

            LogMessageToFile("AHOBPR DAS Export: Sending PDF to DAS for registrant ID = " + questionnaire.RegistrantId);
            try
            {
                Task<HttpResponseMessage> httpRequest = httpClient.SendAsync(requestMessage, HttpCompletionOption.ResponseContentRead, CancellationToken.None);
                HttpResponseMessage httpResponse = httpRequest.Result;
                HttpStatusCode statusCode = httpResponse.StatusCode;
                HttpContent responseContent = httpResponse.Content;
                if (responseContent != null)
                {
                    Task<String> stringContentsTask = responseContent.ReadAsStringAsync();
                    json = stringContentsTask.Result;
                    ProcessResponseFromDas(questionnaire.RegistrantId, json);
                }

                LogMessageToFile("AHOBPR DAS Export: PDF transmission successful.");
            }
            catch (Exception ex)
            {

                LogMessageToFile("AHOBPR DAS Export PDF transmission unsuccessful.  JSON Response:\r\n" + json + "\r\n-----End JSON Response.");
                LogErrorMessage(ex);
                printInnerExceptions(ex, 1);
            }
        }

        private void printInnerExceptions(Exception ex, int count)
        {
            String tabs = "";
            for (int i = 0; i < count; i++)
            {
                tabs += "\t";
            }
            if (ex.InnerException != null)
            {
                LogMessageToFile(tabs + "Inner Exception " + count.ToString());
                LogMessageToFile(tabs + "\t" + ex.InnerException.Message);
                LogMessageToFile(tabs + "\t" + ex.InnerException.StackTrace);
                printInnerExceptions(ex.InnerException, count++);
            }
        }

        private void ProcessResponseFromDas(String registrantId, String json)
        {
            DasResponseValues responseValues = new DasResponseValues(json);

            //Insert a record into the database
            //   using (SqlConnection dbConnection = new SqlConnection(System.Configuration.ConfigurationSettings.AppSettings["SQLServerConnectionString"]))
            using (SqlConnection dbConnection = new SqlConnection(System.Configuration.ConfigurationManager.AppSettings["SQLServerConnectionString"]))
            {
                try
                {
                    dbConnection.Open();
                    if (dbConnection.State == ConnectionState.Open)
                    {
                        LogMessageToFile("The SqlConnection is open.");
                    }
                    else
                    {
                        LogMessageToFile("ERROR: Unabale to open a connection to SQL.");
                        return;
                    }

                    LogMessageToFile("Gathering List Of Registrant IDs that haven't sent PDF's...");
                    string sqlText = "INSERT INTO AHOBPR.INTERFACE_CALL_LOG  \r\n " +
                                    "(REGISTRANT_ID,    INTERFACE_NAME, TIME_OF_CALL,   RETURN_VALUES,  CREATEDBY,UPDATEDBY,CREATED,UPDATED) VALUES \r\n " +
                                    "(@registrantId,    'DAS',          GETDATE(),      @returnValues,  'AHOBPR DAS Export', 'AHOBPR DAS Export', GETDATE(), GETDATE()) ";

                    SqlCommand sqlCmd = new SqlCommand(sqlText, dbConnection);
                    sqlCmd.Parameters.Add(new SqlParameter("@registrantId", registrantId));
                    sqlCmd.Parameters.Add(new SqlParameter("@returnValues", responseValues.Json));
                    try
                    {
                        sqlCmd.ExecuteNonQuery();
                        LogMessageToFile("Logged DAS URI for registrant ID = " + registrantId);
                    }
                    catch (Exception e)
                    {
                        LogErrorMessage(e);
                    }
                }
                catch (Exception ex)
                {
                    LogErrorMessage(ex);
                    LogMessageToFile("AHOBPR DAS Export: Unable to log DAS URI for registrant ID = " + registrantId);
                    return;
                }
            }
        }

        private X509Certificate2 GetCert(String subjectName)
        {
            X509Store store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
            store.Open(OpenFlags.ReadOnly);
            X509Certificate2Collection certificateCollection =
                store.Certificates.Find(X509FindType.FindBySubjectName, subjectName, false);
            if (certificateCollection.Count > 0)
            {
                return certificateCollection[0];
            }
            return null;
        }
    }
}